<?php
/* --------------------------------------------------------------
   ParcelServiceController.php 2020-05-29
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Admin\ParcelService;

use Exception;
use Gambio\Admin\Application\Http\AdminController;
use Gambio\Admin\ParcelService\Interfaces\ParcelServiceService;
use Gambio\Admin\ParcelService\Model\ParcelService;
use Gambio\Admin\ParcelService\Model\ParcelServiceId;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceFilters;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSorting;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSortings;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSqlCriteria;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSqlPagination;
use Slim\Http\Response;
use Slim\Http\ServerRequest;

/**
 * Class ParcelServiceController
 *
 * @package Gambio\Admin\ParcelService
 * @codeCoverageIgnore
 */
class ParcelServiceController extends AdminController
{
    /**
     * @var ParcelServiceService
     */
    private $service;
    
    
    /**
     * ParcelServiceController constructor.
     *
     * @param ParcelServiceService $service
     */
    public function __construct(ParcelServiceService $service)
    {
        $this->service = $service;
    }
    
    
    /**
     * @param ServerRequest $request
     * @param Response      $response
     *
     * @return Response
     */
    public function showOverview(ServerRequest $request, Response $response): Response
    {
        $this->addTemplateDirectory(__DIR__ . '/ui/');
        
        $this->addTemplateTranslations('withdrawals');
        $this->addTemplateTranslations('parcel_services');
        
        $template = $this->render($this->translate('title', 'parcel_services'), __DIR__ . '/ui/overview.html');
        
        return $response->write($template);
    }
    
    
    public function getParcelServices(ServerRequest $request, Response $response): Response
    {
        $_SESSION['coo_page_token']->is_valid($request->getQueryParam('page_token'));
        
        $limit  = (int)$request->getQueryParam('limit', 10);
        $limit  = ($limit <= 0) ? 10 : $limit;
        $offset = (int)$request->getQueryParam('offset', 0);
        $offset = ($offset < 0) ? 0 : $offset;
        $order  = ($request->getQueryParam('order') === 'desc') ? 'desc' : 'asc';
        
        $criteria   = ParcelServiceSqlCriteria::create(ParcelServiceFilters::create(),
                                                       ParcelServiceSortings::create(new ParcelServiceSorting('name',
                                                                                                              $order)));
        $pagination = ParcelServiceSqlPagination::create($limit, $offset);
        
        $parcelServices = $this->service->getAllParcelServices($criteria, $pagination);
        $totalItems     = $this->service->getTotalCountOfParcelServices($criteria);
        
        return $response->withJson([
                                       'success'        => true,
                                       'parcelServices' => $parcelServices,
                                       'totalItems'     => $totalItems,
                                   ]);
    }
    
    
    public function createParcelService(ServerRequest $request, Response $response): Response
    {
        try {
            $data = $request->getParsedBody();
            $_SESSION['coo_page_token']->is_valid($data['page_token'] ?? '');
            
            $data['id']           = null;
            $data['isDefault']    = $data['isDefault'] === 'true';
            $data['descriptions'] = array_map(function ($description) {
                $description['languageId'] = (int)$description['languageId'];
                
                return $description;
            },
                $data['descriptions']);
            $parcelService        = ParcelService::createFromArray($data);
        } catch (Exception $e) {
            return $response->withJson(['success' => false, 'error' => 'Invalid request body.'], 400);
        }
        
        $this->service->storeParcelService($parcelService);
        
        return $response->withJson(['success' => true]);
    }
    
    
    public function updateParcelService(ServerRequest $request, Response $response): Response
    {
        try {
            $data = $request->getParsedBody();
            $_SESSION['coo_page_token']->is_valid($data['page_token'] ?? '');
            
            $data['id']           = (int)$data['id'];
            $data['isDefault']    = $data['isDefault'] === 'true';
            $data['descriptions'] = array_map(function ($description) {
                $description['languageId'] = (int)$description['languageId'];
                
                return $description;
            },
                $data['descriptions']);
            $parcelService        = ParcelService::createFromArray($data);
        } catch (Exception $e) {
            return $response->withJson(['success' => false, 'error' => 'Invalid request body.'], 400);
        }
        
        $this->service->storeParcelService($parcelService);
        
        return $response->withJson(['success' => true]);
    }
    
    
    public function deleteParcelService(ServerRequest $request, Response $response): Response
    {
        $data = $request->getParsedBody();
        $_SESSION['coo_page_token']->is_valid($data['page_token'] ?? '');
        
        $id = (int)$request->getAttribute('id');
        if ($id <= 0) {
            return $response->withJson(['success' => false, 'error' => 'Invalid ID provided.'], 422);
        }
        
        $this->service->deleteParcelService(ParcelServiceId::create($id));
        
        return $response->withJson(['success' => true]);
    }
}